import time as _time
try:
    from micropython import const as _const
    _a = _const(1)
except ImportError:
    def _const(a): return a

# ------------  芯片级常量  ------------
_FT5X06_ADDR        = _const(0x38)

_TD_STATUS_REG      = _const(0x02)   # 触摸点数
_TOUCH_DATA_BASE    = _const(0x03)   # 第 1 个点的 XH
_POINT_STEP         = _const(6)      # 每个点 6 字节
_MAX_POINT          = _const(5)      # 最多 5 点

# ------------  高层封装  ------------
class FT5x06:
    def __init__(self, i2c, address=_FT5X06_ADDR, rst=None):
        self._i2c  = i2c
        self._addr = address
        self._rst  = rst
        self._buf  = bytearray(_MAX_POINT * _POINT_STEP)

    # ---------- API ----------
    def reset(self):
        if self._rst is None:
            return
        self._rst.off()
        _time.sleep_ms(1)
        self._rst.on()
        _time.sleep_ms(200)          # 数据手册建议 > 200 ms

    def get_positions(self):
        """返回 [(x0, y0), (x1, y1), ...]，最多 5 个点"""
        pts = self._i2c.readfrom_mem(self._addr, _TD_STATUS_REG, 1)[0] & 0x0F
        if pts == 0:
            return []

        # 一次把 5 个点的数据全部读出
        self._i2c.readfrom_mem_into(self._addr, _TOUCH_DATA_BASE, self._buf)
        pos = []
        for i in range(min(pts, _MAX_POINT)):
            off = i * _POINT_STEP
            # X/Y 各占 12 位
            x = ((self._buf[off] << 8) | self._buf[off+1]) & 0x0FFF
            y = ((self._buf[off+2] << 8) | self._buf[off+3]) & 0x0FFF
            pos.append((x, y))
        return pos